/*
 * Copyright (c) 1991 AT&T All Rights Reserved
 *	This is unpublished proprietary source code of AT&T
 *	The copyright notice above does not evidence any
 *	actual or intended publication of such source code.
 *
 * Pi Unix driver for Unix System V/386 Version 3.2.3
 *	This file uses /proc to examine running processes.
 *
 * D. A. Kapilow	3/4/91
 */
#include <stdio.h>
#include <signal.h>
#include <pwd.h>
#include <signal.h>
#include <a.out.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/reg.h>
#include <sys/dir.h>
#include <sys/sysmacros.h>
#include <sys/user.h>
#include <sys/immu.h>
#include <sys/procfs.h>
#include <sys/region.h>
#include <sys/proc.h>
#include <sys/trap.h>
#include "dbmonitor/dbhost.h"

#define	bit(x)		(1<<(x-1))
#define	I386_SYSCALL	0x0000000C /* Fix me */
#define	I386_BPT	0xCC
#define REGSIZE		(NGREG * 4)
#define	REG_PC		8
#define	ISCORE(p)	((p)->pid < -1)

/*
 * Data structure private to each process or core dump
 */
struct localproc {
	int		pid;
	int		corefd;
	int		opencnt;
	struct	Unixstate	state;
	struct localproc	*next;
	/* For core dumps */
	int		stabfd;		/* text */
	unsigned long	starttext;
	unsigned long	endtext;
	unsigned long	startdata;
	unsigned long	enddata;
	unsigned long	startstack;
	unsigned long	endstack;
	unsigned long	textoffset;
	unsigned long	dataoffset;
	unsigned long	stackoffset;
	gregset_t	regs;
};

struct localproc *phead;
static char procname[] = "/proc/XXXXX";
static char procbuffer[128];
static int hangpid;
long pi_corepid = -1;
static unsigned long regaddr = -REGSIZE - 4;
static char bkpt_instr = I386_BPT;
static char *pscmds[] = {
	"/bin/ps -u XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX",
	"/bin/ps -a    ",
	"/bin/ps -e    ",
	0
};

/* Routines local to this file */
struct localproc *pi_pidtoproc();
static void state();
static void run();
static int waithandler();
static char *regrw();
static void coreclose();
static char *corerw();
static void reg_sys_to_pi();
static void reg_pi_to_sys();

/* Also use for initialization */
char *pi_osname()
{
	struct passwd *pw;
	static int first = 1;

	if (first) {
		close(2); /* Otherwise error messages from ps goto rtpi */
		open("/dev/null", 2);
		pw = getpwuid(getuid());
		if (pw)
			sprintf(pscmds[0], "/bin/ps -u %s", pw->pw_name);
		else
			sprintf(pscmds[0], "/bin/ps");
		signal(SIGCLD, waithandler);
		first = 0;
	}
	return "System V/386 Release 3.2.3";
}

char	**pi_getpscmds()	{ return pscmds; }
int	pi_getpsfield()		{ return 4; }
long	pi_regaddr()		{ return (long)regaddr; }
long	pi_scratchaddr()	{ return 0xD4; }
int	pi_nsig()		{ return NSIG; }
int	pi_exechangsupported()	{ return 1; }
int	pi_stabfdsupported()	{ return 1; }
int	pi_getppid()		{ return 0; }	/* Fix Me */

long pi_sigmaskinit()
{
	return	bit(SIGILL)|bit(SIGINT)|bit(SIGTRAP)|bit(SIGIOT)|
		bit(SIGEMT)|bit(SIGFPE)|bit(SIGBUS)|bit(SIGSEGV)|
		bit(SIGSYS)|bit(SIGPIPE);
}

struct localproc *pi_open(pid)
{
	register struct localproc *p = phead;
	int wasactive = 0;

	if (pid < 0)
		return 0;
	if (p = pi_pidtoproc(pid)) {
		p->opencnt++;
		return p;
	}
	p = (struct localproc *)malloc(sizeof(struct localproc));
	if (!p)
		return 0;
	sprintf(procname,"/proc/%05d", pid);
	if ((p->corefd = open(procname, 2)) < 0) {
		free(p);
		return 0;
	}
	p->pid = pid;
	p->opencnt = 1;
	p->next = phead;
	phead = p;
	p->state.state = UNIX_ACTIVE;
	return p;
}

void pi_close(p)
struct localproc *p;
{
	register struct localproc *q;

	if (--p->opencnt != 0)
		return;
	if (ISCORE(p))
		coreclose(p);
	else
		close(p->corefd);
	if (p == phead)
		phead = p->next;
	else {
		for (q = phead; q->next != p; q = q->next)
			;
		q->next = p->next;
	}
	free(p);
}

struct localproc *pi_pidtoproc(pid)
{
	register struct localproc *p;

	for (p = phead; p; p = p->next)
		if (p->pid == pid)
			return p;
	return 0;
}

int pi_getsymtabfd(p)
struct localproc *p;
{
	return ioctl(p->corefd, PIOCOPENT, 0);
}

static void run(p, arg)
struct localproc *p;
{
	struct prrun prrun;

	prrun.pr_flags = arg;
	ioctl(p->corefd, PIOCRUN, &prrun);
	p->state.state = UNIX_ACTIVE;
}

char *pi_run(p)
struct localproc *p;
{
	run(p, 0);
	return 0;
}

char *pi_stop(p)
struct localproc *p;
{
	if (ioctl(p->corefd, PIOCSTOP, 0))
		return "PIOCSTOP error";
	return 0;
}

char *pi_destroy(p)
struct localproc *p;
{
	int sig = SIGKILL;
	ioctl(p->corefd, PIOCKILL, &sig);
	state(p);
	p->state.state = UNIX_ERRORED;
	p->state.code = SIGKILL;
	return 0;
}

void pi_getstate(p, s)
struct localproc *p;
struct Unixstate *s;
{
	if (!ISCORE(p))
		state(p);
	*s = p->state;
}

static void state(p)
struct localproc *p;
{
	struct proc pr;
	gregset_t regs;

	if (p->state.state != UNIX_ACTIVE)
		return;
	if (ioctl(p->corefd, PIOCGETPR, &pr) < 0 ) {
		if (p->state.state != UNIX_ERRORED) {
			p->state.state = UNIX_ERRORED;
			p->state.code = NSIG;
		}
		return;
	}
	if (pr.p_stat == SRUN || pr.p_stat == SSLEEP) {
		p->state.state = UNIX_ACTIVE;
	} else if (pr.p_stat == SSTOP) {
		if (!pr.p_cursig)
			p->state.state = UNIX_HALTED;
		else if (pr.p_cursig == SIGTRAP || pr.p_cursig == SIGILL) {
			p->state.state = UNIX_BREAKED;
			/* Fix PC if at a breakpoint */
			ioctl(p->corefd, PIOCGREG, (char *)regs);
			if (regs[TRAPNO] == BPTFLT) {
				regs[EIP] -= sizeof(bkpt_instr);
				ioctl(p->corefd, PIOCSREG, (char *)regs);
			}
			ioctl(p->corefd, PIOCCSIG, 0);
		} else {
			p->state.state = UNIX_PENDING;
			p->state.code = pr.p_cursig;
		}
	} else {
		p->state.state = UNIX_ERRORED;
		p->state.code = NSIG;
	}
}

char *pi_readwrite(p, buf, addr, r, w)
struct localproc *p;
char *buf;
unsigned long addr;
{
	if (ISCORE(p))
		return corerw(p, buf, addr, r, w);
	if (addr >= regaddr)
		return regrw(p, buf, (int)(addr - regaddr), r, w);
	if (lseek(p->corefd, addr, 0) == -1)
		return "lseek on /proc failed";
	if (w) {
		if (write(p->corefd, buf, w) != w)
			return "write on /proc failed";
	} else {
		 if (read(p->corefd, buf, r) != r)
			return "read on /proc failed";
	}
	return 0;
}

static char *regrw(p, buf, offset, r, w)
struct localproc *p;
char *buf;
{
	int wasactive;
	gregset_t regs, piregs;

	if (r) {
		if (ioctl(p->corefd, PIOCGREG, (char*)regs) < 0)
			return "PIOCGREG failed";
		reg_sys_to_pi(regs, piregs);
		bcopy((char*)piregs + offset, buf, r);
		return 0;
	}
	wasactive = 0;
	if (p->state.state == UNIX_ACTIVE) {
		ioctl(p->corefd, PIOCWSTOP, 0);
		state(p);
		if (p->state.state == UNIX_HALTED ||
		    p->state.state == UNIX_ACTIVE)
			wasactive = 1;
	}
	if (ioctl(p->corefd, PIOCGREG, (char*)regs) < 0)
		return "PIOCGREG failed";
	reg_sys_to_pi(regs, piregs);
	bcopy(buf, (char*)piregs + offset, w);
	reg_pi_to_sys(piregs, regs);
	if (ioctl(p->corefd, PIOCSREG, (char*)regs) < 0)
		return "PIOCSREG failed";
	if (wasactive)
		run(p, PRCSIG);
	return 0;
}

char *pi_setbkpt(p, addr)
struct localproc *p;
long addr;
{
	return pi_readwrite(p,(char*)&bkpt_instr, addr, 0, sizeof(bkpt_instr));
}

char *pi_step(p)
struct localproc *p;
{
	run(p, PRSTEP|PRCSIG);
	if (ioctl(p->corefd, PIOCWSTOP, 0))
		return "PIOCWSTOP error";
	p->state.state = UNIX_ACTIVE;
	return 0;
}

static void alarmcatcher()	{}

char *pi_waitstop(p)
struct localproc *p;
{
	char *err = 0;
	void (*save)();

	save = signal(SIGALRM, alarmcatcher);
	alarm(15);
	if (ioctl(p->corefd, PIOCWSTOP, 0) < 0)
		err = "timeout waiting for breakpoint";
	alarm(0);
	signal(SIGALRM, save);
	return err;
}

char *pi_ssig(p, sig)
struct localproc *p;
long sig;
{
	if (ioctl(p->corefd, PIOCKILL, &sig) < 0)
		return "PIOCKILL";
	return 0;
}

char *pi_csig(p)
struct localproc *p;
{
	if (ioctl(p->corefd, PIOCCSIG, 0) < 0)
		return "PIOCCSIG";
	if (p->state.state == UNIX_PENDING)
		p->state.state = UNIX_HALTED;
	return 0;
}

char *pi_sigmask(p, mask)
struct localproc *p;
long mask;
{
	if (ioctl(p->corefd, PIOCSMASK, &mask) < 0)
		return "PIOCSMASK";
	return 0;
}

char *pi_signalname(sig)
long sig;
{
	char *cp;

	switch (sig) {
	case SIGHUP:	cp = "hangup"; break;
	case SIGINT:	cp = "interrupt"; break;
	case SIGQUIT:	cp = "quit"; break;
	case SIGILL:	cp = "illegal instruction"; break;
	case SIGTRAP:	cp = "trace trap"; break;
	case SIGABRT:	cp = "abort"; break;
	case SIGEMT:	cp = "EMT instruction"; break;
	case SIGFPE:	cp = "floating exception"; break;
	case SIGKILL:	cp = "kill"; break;
	case SIGBUS:	cp = "bus error"; break;
	case SIGSEGV:	cp = "segmentation violation"; break;
	case SIGSYS:	cp = "bad system call"; break;
	case SIGPIPE:	cp = "broken pipe"; break;
	case SIGALRM:	cp = "alarm call"; break;
	case SIGTERM:	cp = "terminated"; break;
	case SIGUSR1:	cp = "user signal 1"; break;
	case SIGUSR2:	cp = "user signal 2"; break;
	case SIGCLD:	cp = "child termination"; break;
	case SIGPWR:	cp = "power-fail"; break;
	case SIGWINCH:	cp = "window changed"; break;
	case SIGPOLL:	cp = "pollable event"; break;
	case NSIG:	cp = "cannot determine cause"; break;
	default:
		cp = procbuffer;
		sprintf(procbuffer, "signal %d", sig);
		break;
	}
	return cp;
}

char *pi_proctime(p)
struct localproc *p;
{
	union {
		struct user u;
		char uspace[ctob(USIZE)];
	} u;

	if (ioctl(p->corefd, PIOCGETU, &u) < 0)
		return "";
	procbuffer[0] = 0;
	sprintf(procbuffer, "%d.%02du %d.%02ds",
	  u.u.u_utime/HZ, u.u.u_utime%HZ, u.u.u_stime/HZ, u.u.u_stime%HZ);
	return procbuffer;
}

int pi_atsyscall(p)
struct localproc *p;
{
/*
	long pc, instr; 

	pi_readwrite(p, (char *)&pc, regaddr + REG_PC * 4, sizeof(pc), 0);
	pi_readwrite(p, (char *)&instr, pc, sizeof(instr), 0);
	return (instr == MIPS_SYSCALL);
*/
return 0;
}

char *pi_exechang(p, e)
struct localproc *p;
long e;
{
	if (ioctl(p->corefd, e ? PIOCSEXEC : PIOCREXEC, 0))
		return "PIOCEXEC error";
	return 0;
}

/*
 * This function and the function below are for starting new
 * processes from pi.
 */
int pi_hang(cmd)
char *cmd;
{
	char *argv[30], *cp;
	int i, fd;
	struct localproc *p;
	
	i = strlen(cmd);
	if (++i > sizeof(procbuffer)) {
		i = sizeof(procbuffer) - 1;
		procbuffer[i] = 0;
	}
	bcopy(cmd, procbuffer, i);
	argv[0] = cp = procbuffer;
	for(i = 1;;) {
		while(*cp && *cp != ' ')
			cp++;
		if (!*cp) {
			argv[i] = 0;
			break;
		} else {
			*cp++ = 0;
			while (*cp == ' ')
				cp++;
			if (*cp)
				argv[i++] = cp;
		}
	}
	sprintf(procname, "/proc/%05d", getpid());
	if ((fd = open(procname, 2)) < 0)
		return 0;
	ioctl(fd, PIOCSEXEC, 0);
	hangpid = fork();
	if (!hangpid){
		for (fd = 0; fd < NOFILE; ++fd)
			close(fd);
		open("/dev/null", 2);
		dup2(0, 1);
		dup2(0, 2);
		setpgrp(0, getpid());
		execvp(argv[0], argv);
		exit(0);
	}
	ioctl(fd, PIOCREXEC, 0);
	close(fd);
	if (hangpid < 0 || !(p = pi_open(hangpid)))
		return 0;
	p->opencnt = 0;
	pi_waitstop(p);
	return hangpid;
}

static int waithandler()
{
	int pid, cursig;
	int wstat;
	struct localproc *p;

	pid = wait(&wstat);
	signal(SIGCLD, waithandler);
	if (pid < 0 || !(p = pi_pidtoproc(pid)))
		return;
	if ((wstat&0377) == 0177) {
		cursig = (wstat & 0xFF) >> 8;
		if (cursig == SIGTRAP)
			p->state.state = UNIX_BREAKED;
		else {
			p->state.state = UNIX_PENDING;
			p->state.code = cursig;
		}
	} else {
		p->state.state = UNIX_ERRORED;
		p->state.code = wstat & 0xFFFF;
	}
}

/*
 * The functions below are for debugging core dumps
 */
struct localproc *pi_coreopen(corep, symtabp)
char *corep, *symtabp;
{
	register struct localproc *p;
	int symtabfd, corefd = -1;
	int mode;
	struct user u;
	struct filehdr fh;
	gregset_t regs;

	if ((symtabfd = open(symtabp, 0)) < 0)
		return 0;
	for (mode = 2; corefd < 0 && mode >= 0; mode--)
		corefd = open(corep, mode);
	if (corefd < 0) {
		close(symtabfd);
		return 0;
	}
	if (read(corefd, (char*)&u, sizeof(u)) != sizeof(u) ||
	    u.u_exdata.ux_mag != 0413 ||
	    read(symtabfd, (char*)&fh, sizeof(fh)) != sizeof(fh) ||
	    fh.f_magic != I386MAGIC ||
	    !(p = (struct localproc *)malloc(sizeof(struct localproc)))) {
		close(corefd);
		close(symtabfd);
		return 0;
	}
	p = (struct localproc *)malloc(sizeof(struct localproc));
	if (!p) {
		close(corefd);
		close(symtabfd);
		return 0;
	}
	p->pid = --pi_corepid;
	p->opencnt = 1;
	p->next = phead;
	phead = p;
	p->corefd = corefd;
	p->stabfd = symtabfd;

	p->starttext = u.u_exdata.ux_txtorg;
	p->endtext = p->starttext + u.u_exdata.ux_tsize;
	p->textoffset = u.u_exdata.ux_toffset;
	p->startdata = u.u_exdata.ux_datorg;
	p->enddata = p->startdata + ctob(u.u_dsize);
	p->dataoffset = ctob(USIZE);
	p->startstack = u.u_sub;
	p->endstack = p->startstack + ctob(u.u_ssize);
	p->stackoffset = p->dataoffset + ctob(u.u_dsize);
	if (lseek(p->corefd, (unsigned)u.u_ar0 - UVUBLK, 0) == -1 ||
	    read(p->corefd, (char*)regs, sizeof(regs)) != sizeof(regs)) {
			pi_close(p);
			return 0;
	}
	reg_sys_to_pi(regs, p->regs);
	p->state.state = UNIX_PENDING;
	p->state.code = u.u_sysabort;
	return p;
}

static void coreclose(p)
struct localproc *p;
{
	close(p->corefd);
	close(p->stabfd);
}

static char *corerw(p, buf, addr, r, w)
struct localproc *p;
char *buf;
unsigned long addr;
{
	int fd;

	if (p->starttext <= addr && addr < p->endtext) {
		fd = p->stabfd;
		addr -= p->starttext - p->textoffset;
	}
	else if (p->startdata <= addr && addr < p->enddata) {
		fd = p->corefd;
		addr -= p->startdata - p->dataoffset;
	}
	else if (p->startstack <= addr && addr < p->endstack) {
		fd = p->corefd;
		addr -= p->startstack - p->stackoffset;
	}
	else if (regaddr <= addr && addr < (regaddr+REGSIZE)) {
		addr -= regaddr;
		if (r)
			bcopy((char*)p->regs + addr, buf, r);
		else {
			if ((addr + w) > REGSIZE)
				w = REGSIZE - addr;
			bcopy(buf, (char*)p->regs + addr, w);
		}
		return 0;
	}
	else
		return "corerw:invalid address";
	if (lseek(fd, addr, 0) == -1)
		return "corerw:lseek failed";
	if (w) {
		if (write(fd, buf, w) != w)
			return "corerw:write error";
	} else {
		if (read(fd, buf, r) != r)
			return "corerw:read error";
	}
	return 0;
}

static void reg_sys_to_pi(sys, pi)
register int *sys, *pi;
{
	*pi++ = sys[EAX];
	*pi++ = sys[ECX];
	*pi++ = sys[EDX];
	*pi++ = sys[EBX];
	*pi++ = sys[UESP];
	*pi++ = sys[EBP];
	*pi++ = sys[ESI];
	*pi++ = sys[EDI];
	*pi++ = sys[EIP];
	*pi++ = sys[EFL];
	*pi++ = sys[DS];
	*pi++ = sys[ES];
	*pi++ = sys[FS];
	*pi++ = sys[GS];
	*pi++ = sys[SS];
	*pi++ = sys[CS];
	*pi++ = sys[TRAPNO];
	*pi++ = sys[ERR];
	*pi = sys[ESP];
}

static void reg_pi_to_sys(pi, sys)
register int *pi, *sys;
{
	sys[EAX] = *pi++;
	sys[ECX] = *pi++;
	sys[EDX] = *pi++;
	sys[EBX] = *pi++;
	sys[UESP] = *pi++;
	sys[EBP] = *pi++;
	sys[ESI] = *pi++;
	sys[EDI] = *pi++;
	sys[EIP] = *pi++;
	sys[EFL] = *pi++;
	sys[DS] = *pi++;
	sys[ES] = *pi++;
	sys[FS] = *pi++;
	sys[GS] = *pi++;
	sys[SS] = *pi++;
	sys[CS] = *pi++;
	sys[TRAPNO] = *pi++;
	sys[ERR] = *pi++;
	sys[ESP] = *pi;
}
